import warnings


def iou(gt_bb, pred_bb):
    '''
    :param gt_bb: ground truth bounding box
    :param pred_bb: predicted bounding box
    '''
    def box_valid(bb):
        return len(bb) == 4 and gt_bb[0] <= gt_bb[2] and gt_bb[1] <= gt_bb[3]

    if not box_valid(gt_bb):
        raise ValueError("gt_bb is not a valid bounding box")

    if not box_valid(pred_bb):
        raise ValueError("pred_bb is not a valid bounding box")

    def box_size(bb):
        width = bb[2] - bb[0]
        height = bb[3] - bb[1]
        if width < 0 or height < 0:
            return 0

        return width * height

    inter_bb = [
        max(gt_bb[0], pred_bb[0]),
        max(gt_bb[1], pred_bb[1]),
        min(gt_bb[2], pred_bb[2]),
        min(gt_bb[3], pred_bb[3]),
    ]

    inter_size = box_size(inter_bb)

    if inter_size == 0.0:
        return 0.0

    return float(inter_size) / float(
        box_size(gt_bb) + box_size(pred_bb) - inter_size)


def check_boundingbox(gt_bb, pred_bb):
    """
    Checks if a single predicted bounding box matches a ground truth bounding box.
    """
    return iou(gt_bb, pred_bb) > 0.8


def check_boundingbox_list(gt_bbs, pred_bbs):
    """
    Given a list of ground truth and predicted bounding boxes caculates how many boxes match.

    :returns: Tuple of number of ground-truth boxes matching and of predicted boxes matching a ground truth box.
    """
    def count_matches(bbs, bbs2):
        matches = 0

        for bb in bbs:
            for bb2 in bbs2:
                if check_boundingbox(bb, bb2):
                    matches += 1
                    break

        return matches

    matches_gt = count_matches(gt_bbs, pred_bbs)
    matches_pred = count_matches(pred_bbs, gt_bbs)

    return matches_gt, matches_pred


def score(gt, pred):
    """
    :param gt: ground truth bounding boxes given as dict with type as key and values as list of bounding boxes
    :param pred: predicted bounding boxes given in the same way as gt
    """

    if not isinstance(gt, dict):
        raise ValueError

    if not isinstance(pred, dict):
        raise ValueError

    # all occuring keys
    if not set(pred.keys()).issubset(set(gt.keys())):
        warnings.warn("Prediction has extra keys not present in ground truth")

    res = {}
    score_all = 0.0

    for key in gt.keys():
        gt_bbs = gt.get(key, [])
        pred_bbs = pred.get(key, [])

        matched_gt, matched_pred = check_boundingbox_list(gt_bbs, pred_bbs)

        unmatched_pred = len(pred_bbs) - matched_pred

        if gt_bbs:
            score = matched_gt / float(len(gt_bbs))
        else:
            score = 1.0

        score = score * (0.75**unmatched_pred)

        res[key + "_matched_gt"] = matched_gt
        res[key + "_matched_pred"] = matched_pred
        res[key + "_total_pred"] = len(pred_bbs)
        res[key + "_total_gt"] = len(gt_bbs)
        res[key + "_score"] = score
        score_all = score_all + score

    res["score"] = score_all / len(gt)

    return res
