#!/usr/bin/env python

import sys
import json
import os
import argparse

from PIL import Image, ImageDraw

COLORS = {
    "name": "#173F5F",
    "address": "#20639B",
    "phone": "#3CAEA3",
    "signature": "#F6D55C",
    "handwritten": "#ED553B"
}


def process_annotation(s, out_dir, invoice_dir, box_type):
    ann = json.loads(s)

    in_fn = os.path.join(invoice_dir, ann["id"])
    out_fn = os.path.join(out_dir, ann["id"])
    bounding_boxes = ann["bounding_boxes"]

    im = Image.open(in_fn)
    im = im.convert("RGB")
    width, height = im.size
    draw = ImageDraw.Draw(im)

    for bb in bounding_boxes:
        x0 = bb["x0"] * width
        y0 = bb["y0"] * height
        x1 = bb["x1"] * width
        y1 = bb["y1"] * height

        rec_args = {
            "xy": ((x0, y0), (x1, y1)),
            box_type: COLORS.get(bb["label"], "#FF0000")
        }

        draw.rectangle(**rec_args)

    im.save(out_fn)


def main(input_file, out_dir, invoice_dir, box_type):
    if out_dir is None:
        out_dir = "out"

    if not os.path.isdir(out_dir):
        os.makedirs(out_dir)

    for s in input_file:
        s = s.strip()
        process_annotation(s, out_dir, invoice_dir, box_type)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=
        'Sample for Task 2. This script anonymize invoices by replacing them with colored boxes.'
    )
    parser.add_argument("--invoice-dir", type=str, default="invoices")
    parser.add_argument("--box",
                        type=str,
                        choices=["fill", "outline"],
                        default="fill")
    parser.add_argument("input",
                        type=argparse.FileType("r"),
                        help="Filename annotations to read.")
    parser.add_argument("outdir",
                        type=str,
                        nargs='?',
                        help="Directory where to write the resulting images.")

    args = parser.parse_args()

    main(args.input, args.outdir, args.invoice_dir, args.box)
